using System;
using System.Drawing;
using System.Text;
using System.IO;
using System.ComponentModel.Design;
using System.Collections.Specialized;
using System.Windows.Forms;
using System.Reflection;

using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.Handles;
using Borland.Eco.Handles.Design;
using Borland.Eco.Persistence.Design;

namespace Borland.Eco.Examples.EcoSpaceDesignerPlugins
{
	enum Position
	{
		Start,
		Stop
	}

	/// <summary>
	/// HtmlReportTool implements IEcoSpaceDesignerTool by inheritance
	/// </summary>
	[ToolboxBitmap(typeof(HtmlReportTool), "Borland.Eco.Examples.EcoSpaceDesignerPlugins.HtmlReportTool.bmp")]
	public class HtmlReportTool: DBToolBase
	{
		/// <summary>
		/// Registration method, called automatically by the IDE
		/// required to register the tool.
		///
		/// Create a new instance of the tool and make it known to the registry of tools.
		/// </summary>
		public static void IDERegister()
		{
			EcoSpaceDesignerToolbox.AddTool(new HtmlReportTool());
		}
		/// <summary>
		/// Implementation of IEcoSpaceDesignerTool.GetHint.
		/// This is the flyover hint for the tool
		/// </summary>
		public override string GetHint()
		{
			return "HTML Report of Model";
		}
		private string UMLVisibility(VisibilityKind visibility)
		{
			switch (visibility)
			{
				case VisibilityKind.Private_: return "-";
				case VisibilityKind.Protected_: return "#";
				case VisibilityKind.Public_: return "+";
				default: return "+";
			}
		}

		private string UMLDerived(bool derived)
		{
			if (derived)
				return "/";
			else
				return string.Empty;
		}
		private string Params(IMethod method)
		{
			StringBuilder Result = new StringBuilder();
			for (int i = 0; i < method.Parameters.Count - 1; i++)
			{
				Result.AppendFormat("{0} {1}", method.Parameters[i].Type.Name, method.Parameters[i].Name);
				if (i < method.Parameters.Count - 2)
					Result.Append(", ");
			}
			return Result.ToString();
		}
		private string ReturnType(IMethod method)
		{
			if (method.ReturnType != null)
				return ": " + method.ReturnType.Name;
			else
				return string.Empty;
		}

		private string HtmlForAbstract(bool isAbstract, Position marker)
		{
			if (!isAbstract) return string.Empty;
			if (marker == Position.Start)
				return "<I>";
			else
				return "</I>";
		}
		private DefaultEcoSpace ecoSpace;
		private ITypeSystemService typeSystemService;
		private ITypeResolutionService typeResolutionService;
		/// <summary>
		/// Implementation of IEcoSpaceDesignerTool.Execute
		/// Execute is called when the GUI widget (button) is invoked.
		/// </summary>
		public override void Execute(ExecuteArgs executeArgs)
		{
			this.ecoSpace = executeArgs.EcoSpace;
			this.typeResolutionService = executeArgs.TypeResolutionService;

			this.typeSystemService = IdeHelper.GetTypeSystemService(executeArgs.Component.Site.Name, true, typeResolutionService);
			IClassCollection allClasses = typeSystemService.TypeSystem.AllClasses;

			StringBuilder html = new StringBuilder();

			html.Append("<html><body>");
			html.AppendFormat("<h1>{0}</h1>", typeSystemService.TypeSystem.Name);

			html.Append("<a name=\"top\">List of Classes<p><ul>");

			for (int i = 0; i < allClasses.Count; i++)
			{
				html.AppendFormat("{1}<a name=\"top{0}\"><A HREF=\"#{0}\">{0}</A>{2}<BR>", allClasses[i].Name,
					HtmlForAbstract(allClasses[i].EcoClassifier.IsAbstract, Position.Start),
					HtmlForAbstract(allClasses[i].EcoClassifier.IsAbstract, Position.Stop));
			}
			html.Append("</ul><hr>");

			for (int i = 0; i < allClasses.Count; i++)
			{
				StringBuilder attributes = new StringBuilder();
				StringBuilder associationEnds = new StringBuilder();
				StringBuilder operations = new StringBuilder();

				for (int m = 0; m < allClasses[i].Features.Count; m++)
				{
					IFeature feature = allClasses[i].Features[m] as IFeature;
					switch (feature.FeatureType)
					{
						case FeatureType.Attribute:
							IStructuralFeature attribute = feature as IStructuralFeature;
							attributes.AppendFormat("{0}{1}{2}: {3}<br>", UMLDerived(attribute.EcoStructuralFeature.IsDerived), UMLVisibility(attribute.Visibility), attribute.Name, attribute.Type_.Name);
							break;

						case FeatureType.AssociationEnd:
							IStructuralFeature associationEnd = feature as IStructuralFeature;
							associationEnds.AppendFormat("{0}{1}{2}: <a href=\"#{3}\">{3}</a><br>", UMLDerived(associationEnd.EcoStructuralFeature.IsDerived), UMLVisibility(associationEnd.Visibility), associationEnd.Name, associationEnd.Type_.Name);
							break;

						case FeatureType.Method:
							IMethod method = feature as IMethod;
							operations.AppendFormat("{0}{1}({2}){3}<br>", UMLVisibility(method.Visibility), method.Name, Params(method), ReturnType(method));
							break;
					}
				}

/** Want to look at non-modelled operations? Uncomment this block and do it :) **/

/*
				// Extract method information via reflection
				System.Type t = typeResolutionService.GetType(allClasses[i].Name);
				if (t != null)
				{
					MethodInfo[] methods = t.GetMethods();

					foreach (MethodInfo method in methods)
					{
						if (!method.IsSpecialName && method.ReflectedType == t) // Skip accessors and ancestor methods
						{
							if (method.IsPrivate) operations.Append("-");
							if (method.IsFamily || method.IsFamilyAndAssembly || method.IsFamilyOrAssembly) operations.Append("#");
							if (method.IsPublic) operations.Append("+");

							operations.AppendFormat("{0}{1} {2}(", HtmlForAbstract(method.IsAbstract, Position.Start),
								method.ReturnType.ToString(), method.Name);

							ParameterInfo[] parameters = method.GetParameters();
							int paramCount = 0;
							foreach (ParameterInfo parameter in parameters)
							{
								operations.AppendFormat("{0} {1}, ", parameter.ParameterType.ToString(), parameter.Name);
								paramCount++;
							}
							if (paramCount > 0)
								operations.Remove(operations.Length - 2, 2);
							operations.AppendFormat("){0}<br>", HtmlForAbstract(method.IsAbstract, Position.Stop));
						}
					}
				}
*/

				html.Append(string.Empty);
				html.Append(System.String.Format("<H2>{1}<A NAME=\"{0}\">{0}{2}</h2><P>", allClasses[i].Name,
					HtmlForAbstract(allClasses[i].EcoClassifier.IsAbstract, Position.Start),
					HtmlForAbstract(allClasses[i].EcoClassifier.IsAbstract, Position.Stop)));
				html.Append(string.Empty);


				html.Append("<H3>Attributes</H3>");
				html.Append(attributes.ToString());

				html.Append("<H3>Associations</H3>");
				html.Append(associationEnds.ToString());

				html.Append("<H3>Operations</H3>");
				html.Append(operations.ToString());

				html.Append("<H4>Back to <A href=\"#top\">class list</h4><P>");
				html.Append("<hr>");
			}


			html.Append("</body></html>");

			SaveFileDialog fsd = new SaveFileDialog();

			fsd.DefaultExt = "html";
			if (fsd.ShowDialog() == DialogResult.OK)
			{
				using (StreamWriter sw = new StreamWriter(fsd.FileName))
				{
					// Add some text to the file.
					sw.WriteLine(html.ToString());
				}
			}
		}
	}
}